<?php
session_start();
include 'assets/config/connect.php';

$error = ''; // Initialize error variable

// Handle form submission
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    // Validate and process form data
    $sku = $_POST['sku'];
    $name = $_POST['name'];
    $description = $_POST['description'];
    $price = $_POST['price'];
    $stock_quantity = $_POST['stock_quantity'];
    $weight_value = $_POST['weight_value'];
    $weight_unit = $_POST['weight_unit'];
    
    // Generate a unique SKU if not provided
    if (empty($sku)) {
        $sku = "GYM" . uniqid();
    }
    
    // Check if SKU already exists
    $check_sku = $conn->prepare("SELECT id FROM products WHERE sku = ?");
    $check_sku->bind_param("s", $sku);
    $check_sku->execute();
    $check_sku->store_result();
    
    if ($check_sku->num_rows > 0) {
        $error = "SKU already exists. Please use a different SKU.";
    } else {
        // Insert product into database
        $stmt = $conn->prepare("INSERT INTO products (sku, name, description, price, stock_quantity, weight_value, weight_unit) VALUES (?, ?, ?, ?, ?, ?, ?)");
        $stmt->bind_param("sssdisd", $sku, $name, $description, $price, $stock_quantity, $weight_value, $weight_unit);
        
        if ($stmt->execute()) {
            $product_id = $stmt->insert_id;
            
            // Handle image uploads
            if (!empty($_FILES['product_images']['name'][0])) {
                $upload_dir = "assets/uploads/products/";
                if (!is_dir($upload_dir)) {
                    mkdir($upload_dir, 0777, true);
                }
                
                $uploaded_images = 0;
                $max_images = 5;
                
                foreach ($_FILES['product_images']['tmp_name'] as $key => $tmp_name) {
                    if ($uploaded_images >= $max_images) break;
                    
                    $file_name = $_FILES['product_images']['name'][$key];
                    $file_size = $_FILES['product_images']['size'][$key];
                    $file_tmp = $_FILES['product_images']['tmp_name'][$key];
                    $file_type = $_FILES['product_images']['type'][$key];
                    
                    // Check file size (max 2MB)
                    if ($file_size > 2097152) {
                        $error .= "File $file_name is too large. Maximum size is 2MB.<br>";
                        continue;
                    }
                    
                    // Check if file is an image
                    $allowed_types = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif'];
                    if (!in_array($file_type, $allowed_types)) {
                        $error .= "File $file_name is not a valid image type.<br>";
                        continue;
                    }
                    
                    // Generate unique filename
                    $file_ext = pathinfo($file_name, PATHINFO_EXTENSION);
                    $new_filename = "product_" . $product_id . "_" . time() . "_" . $key . "." . $file_ext;
                    $file_path = $upload_dir . $new_filename;
                    
                    // Move uploaded file
                    if (move_uploaded_file($file_tmp, $file_path)) {
                        // Set first image as primary
                        $is_primary = ($key == 0) ? 1 : 0;
                        
                        // Save image info to database
                        $img_stmt = $conn->prepare("INSERT INTO product_images (product_id, image_path, is_primary) VALUES (?, ?, ?)");
                        $img_stmt->bind_param("isi", $product_id, $file_path, $is_primary);
                        $img_stmt->execute();
                        $img_stmt->close();
                        
                        $uploaded_images++;
                    } else {
                        $error .= "Failed to upload $file_name.<br>";
                    }
                }
            }
            
            // Redirect to success page or show success message
            $_SESSION['success_message'] = "Product added successfully!";
            header("Location: product-list.php");
            exit();
        } else {
            $error = "Error adding product: " . $conn->error;
        }
        
        $stmt->close();
    }
    
    $check_sku->close();
}
?>

<!doctype html>
<html lang="en">

<head>
    <meta charset="utf-8" />
    <title>Gym - Add Product</title>
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta content="Premium Multipurpose Admin & Dashboard Template" name="description" />
    <meta content="Themesbrand" name="author" />
    <!-- App favicon -->
    <link rel="shortcut icon" href="assets/images/logo-sm.png">

    <!-- Bootstrap Css -->
    <link href="assets/css/bootstrap.min.css" id="bootstrap-style" rel="stylesheet" type="text/css" />
    <!-- Icons Css -->
    <link href="assets/css/icons.min.css" rel="stylesheet" type="text/css" />
    <!-- App Css-->
    <link href="assets/css/app.min.css" id="app-style" rel="stylesheet" type="text/css" />
    
    <style>
        .error-message {
            color: #dc3545;
            font-size: 0.875em;
            margin-top: 0.25rem;
        }

        .is-invalid {
            border-color: #dc3545 !important;
        }
        
        .image-preview-container {
            display: flex;
            flex-wrap: wrap;
            gap: 10px;
            margin-top: 10px;
        }
        
        .image-preview {
            width: 100px;
            height: 100px;
            border: 1px solid #ddd;
            border-radius: 4px;
            padding: 4px;
            position: relative;
        }
        
        .image-preview img {
            width: 100%;
            height: 100%;
            object-fit: cover;
        }
        
        .remove-image {
            position: absolute;
            top: -10px;
            right: -10px;
            background: #dc3545;
            color: white;
            border-radius: 50%;
            width: 20px;
            height: 20px;
            line-height: 20px;
            text-align: center;
            cursor: pointer;
            font-size: 12px;
        }
        
        /* Rich Text Editor Styles */
        .rich-text-editor {
            border: 1px solid #ced4da;
            border-radius: 0.25rem;
        }
        
        .editor-toolbar {
            padding: 0.5rem;
            border-bottom: 1px solid #dee2e6;
            background-color: #f8f9fa;
            display: flex;
            flex-wrap: wrap;
            gap: 0.25rem;
        }
        
        .editor-content {
            min-height: 200px;
            padding: 0.75rem;
            overflow-y: auto;
        }
        
        .editor-content:focus {
            outline: none;
        }
        
        .font-selector {
            width: 120px;
        }
        
        .font-size-selector {
            width: 80px;
        }
        
        .color-selector {
            width: 50px;
            padding: 0.25rem;
        }
        
        .editor-btn-group {
            display: flex;
            gap: 0.1rem;
        }
        
        .editor-btn {
            padding: 0.25rem 0.5rem;
            background: white;
            border: 1px solid #dee2e6;
            border-radius: 0.2rem;
            cursor: pointer;
        }
        
        .editor-btn:hover {
            background-color: #e9ecef;
        }
        
        .editor-btn.active {
            background-color: #007bff;
            color: white;
            border-color: #007bff;
        }
        
        .divider {
            width: 1px;
            background-color: #dee2e6;
            margin: 0 0.5rem;
            align-self: stretch;
        }
    </style>
</head>

<body data-sidebar="dark">
    <!-- Begin page -->
    <div id="layout-wrapper">
        <?php include 'assets/config/header.php' ?>
        <?php include 'assets/config/sidebar.php' ?>

        <div class="main-content">
            <div class="page-content">
                <div class="container-fluid">
                    <!-- start page title -->
                    <div class="row">
                        <div class="col-sm-6">
                            <div class="page-title-box">
                                <h4>Products</h4>
                                <ol class="breadcrumb m-0">
                                    <li class="breadcrumb-item active">Add Product</li>
                                </ol>
                            </div>
                        </div>
                    </div>
                    <!-- end page title -->

                    <div class="row">
                        <div class="col-12">
                            <div class="card">
                                <div class="card-body">
                                    <h4 class="card-title">Add New Product</h4>
                                    <p class="card-title-desc">Fill all product details carefully</p>
                                    
                                    <?php if (!empty($error)): ?>
                                    <div class="alert alert-danger" role="alert">
                                        <?php echo $error; ?>
                                    </div>
                                    <?php endif; ?>
                                    
                                    <form id="productForm" method="POST" action="" enctype="multipart/form-data">
                                        <div class="row">
                                            <!-- SKU -->
                                            <div class="col-md-6 mb-3">
                                                <label class="form-label">SKU (Leave blank to auto-generate)</label>
                                                <input type="text" class="form-control" name="sku" placeholder="e.g., PROD001">
                                            </div>

                                            <!-- Product Name -->
                                            <div class="col-md-6 mb-3">
                                                <label class="form-label">Product Name</label>
                                                <input type="text" class="form-control" name="name" required>
                                            </div>

                                            <!-- Price -->
                                            <div class="col-md-6 mb-3">
                                                <label class="form-label">Price ($)</label>
                                                <input type="number" class="form-control" name="price" step="0.01" min="0" required>
                                            </div>

                                            <!-- Stock Quantity -->
                                            <div class="col-md-6 mb-3">
                                                <label class="form-label">Stock Quantity</label>
                                                <input type="number" class="form-control" name="stock_quantity" min="0" required>
                                            </div>

                                            <!-- Weight -->
                                            <div class="col-md-4 mb-3">
                                                <label class="form-label">Weight Value</label>
                                                <input type="number" class="form-control" name="weight_value" step="0.01" min="0" required>
                                            </div>

                                            <!-- Weight Unit -->
                                            <div class="col-md-2 mb-3">
                                                <label class="form-label">Unit</label>
                                                <select class="form-control" name="weight_unit" required>
                                                    <option value="g">g</option>
                                                    <option value="kg">kg</option>
                                                    <option value="lb">lb</option>
                                                    <option value="oz">oz</option>
                                                </select>
                                            </div>

                                            <!-- Product Images -->
                                            <div class="col-md-12 mb-3">
                                                <label class="form-label">Product Images (Max 5 images, 2MB each)</label>
                                                <input type="file" class="form-control" name="product_images[]" multiple accept="image/*" onchange="previewImages(this)">
                                                <div class="image-preview-container" id="imagePreviewContainer"></div>
                                            </div>

                                            <!-- Description -->
                                            <div class="col-md-12 mb-3">
                                                <label class="form-label">Description</label>
                                                <div class="rich-text-editor">
                                                    <div class="editor-toolbar">
                                                        <div class="editor-btn-group">
                                                            <select class="form-control form-control-sm font-selector" id="fontSelector">
                                                                <option value="Arial">Arial</option>
                                                                <option value="Helvetica">Helvetica</option>
                                                                <option value="Times New Roman">Times New Roman</option>
                                                                <option value="Courier New">Courier New</option>
                                                                <option value="Verdana">Verdana</option>
                                                                <option value="Georgia">Georgia</option>
                                                            </select>
                                                        </div>
                                                        
                                                        <div class="editor-btn-group">
                                                            <select class="form-control form-control-sm font-size-selector" id="fontSizeSelector">
                                                                <option value="1">Small</option>
                                                                <option value="3" selected>Normal</option>
                                                                <option value="5">Large</option>
                                                                <option value="7">X-Large</option>
                                                            </select>
                                                        </div>
                                                        
                                                        <div class="editor-btn-group">
                                                            <button type="button" class="editor-btn" data-command="bold" title="Bold"><strong>B</strong></button>
                                                            <button type="button" class="editor-btn" data-command="italic" title="Italic"><em>I</em></button>
                                                            <button type="button" class="editor-btn" data-command="underline" title="Underline"><u>U</u></button>
                                                        </div>
                                                        
                                                        <div class="editor-btn-group">
                                                            <button type="button" class="editor-btn" data-command="justifyLeft" title="Align Left"><i class="fas fa-align-left"></i></button>
                                                            <button type="button" class="editor-btn" data-command="justifyCenter" title="Align Center"><i class="fas fa-align-center"></i></button>
                                                            <button type="button" class="editor-btn" data-command="justifyRight" title="Align Right"><i class="fas fa-align-right"></i></button>
                                                        </div>
                                                        
                                                        <div class="editor-btn-group">
                                                            <button type="button" class="editor-btn" data-command="insertUnorderedList" title="Bullet List"><i class="fas fa-list-ul"></i></button>
                                                            <button type="button" class="editor-btn" data-command="insertOrderedList" title="Numbered List"><i class="fas fa-list-ol"></i></button>
                                                        </div>
                                                        
                                                        <div class="editor-btn-group">
                                                            <input type="color" class="form-control form-control-sm color-selector" id="colorSelector" value="#000000" title="Text Color">
                                                        </div>
                                                        
                                                        <div class="editor-btn-group">
                                                            <button type="button" class="editor-btn" data-command="formatBlock" data-value="h1" title="Heading 1">H1</button>
                                                            <button type="button" class="editor-btn" data-command="formatBlock" data-value="h2" title="Heading 2">H2</button>
                                                            <button type="button" class="editor-btn" data-command="formatBlock" data-value="h3" title="Heading 3">H3</button>
                                                        </div>
                                                        
                                                        <div class="editor-btn-group">
                                                            <button type="button" class="editor-btn" data-command="createLink" title="Insert Link"><i class="fas fa-link"></i></button>
                                                            <button type="button" class="editor-btn" data-command="unlink" title="Remove Link"><i class="fas fa-unlink"></i></button>
                                                        </div>
                                                        
                                                        <div class="editor-btn-group">
                                                            <button type="button" class="editor-btn" data-command="removeFormat" title="Clear Formatting"><i class="fas fa-eraser"></i></button>
                                                        </div>
                                                    </div>
                                                    <div class="editor-content form-control" id="editor" contenteditable="true"></div>
                                                    <textarea id="description" name="description" style="display:none;"></textarea>
                                                </div>
                                            </div>

                                            <!-- Submit Button -->
                                            <div class="col-md-12 text-center mt-3">
                                                <button type="submit" class="btn btn-primary">Add Product</button>
                                                <button type="reset" class="btn btn-secondary">Reset</button>
                                            </div>
                                        </div>
                                    </form>
                                </div>
                            </div>
                        </div>
                    </div>

                </div> <!-- container-fluid -->
            </div>
            <!-- End Page-content -->
        </div>
        <!-- end main content-->
    </div>
    <!-- END layout-wrapper -->

    <!-- JAVASCRIPT -->
    <script src="assets/libs/jquery/jquery.min.js"></script>
    <script src="assets/libs/bootstrap/js/bootstrap.bundle.min.js"></script>
    <script src="assets/libs/metismenu/metisMenu.min.js"></script>
    <script src="assets/libs/simplebar/simplebar.min.js"></script>
    <script src="assets/libs/node-waves/waves.min.js"></script>
    <script src="assets/libs/jquery-sparkline/jquery.sparkline.min.js"></script>
    
    <!-- Font Awesome for icons -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.3/css/all.min.css">
    
    <script src="assets/js/app.js"></script>
    
    <script>
        // Initialize Bootstrap rich text editor
        document.addEventListener('DOMContentLoaded', function() {
            const editor = document.getElementById('editor');
            const descriptionField = document.getElementById('description');
            const fontSelector = document.getElementById('fontSelector');
            const fontSizeSelector = document.getElementById('fontSizeSelector');
            const colorSelector = document.getElementById('colorSelector');
            
            // Function to update the hidden textarea with editor content
            function updateDescriptionField() {
                descriptionField.value = editor.innerHTML;
            }
            
            // Update the hidden field on input
            editor.addEventListener('input', updateDescriptionField);
            
            // Toolbar button functionality
            document.querySelectorAll('.editor-btn').forEach(button => {
                button.addEventListener('click', function() {
                    const command = this.dataset.command;
                    const value = this.dataset.value;
                    
                    // Handle special commands
                    if (command === 'createLink') {
                        const url = prompt('Enter the URL:');
                        if (url) {
                            document.execCommand(command, false, url);
                        }
                    } else if (value) {
                        document.execCommand(command, false, value);
                    } else {
                        document.execCommand(command, false, null);
                    }
                    
                    // Update the hidden field
                    updateDescriptionField();
                    
                    // Maintain focus on editor
                    editor.focus();
                });
            });
            
            // Font selector
            fontSelector.addEventListener('change', function() {
                document.execCommand('fontName', false, this.value);
                updateDescriptionField();
                editor.focus();
            });
            
            // Font size selector
            fontSizeSelector.addEventListener('change', function() {
                document.execCommand('fontSize', false, this.value);
                updateDescriptionField();
                editor.focus();
            });
            
            // Color selector
            colorSelector.addEventListener('change', function() {
                document.execCommand('foreColor', false, this.value);
                updateDescriptionField();
                editor.focus();
            });
            
            // Handle form submission
            document.getElementById('productForm').addEventListener('submit', function() {
                updateDescriptionField();
            });
            
            // Set default content for the editor
            editor.innerHTML = '<p>Enter product description here...</p>';
            updateDescriptionField();
        });
        
        // Image preview functionality
        function previewImages(input) {
            const previewContainer = document.getElementById('imagePreviewContainer');
            previewContainer.innerHTML = '';
            
            if (input.files && input.files.length > 0) {
                const files = Array.from(input.files).slice(0, 5); // Limit to 5 files
                
                files.forEach((file, index) => {
                    if (file.size > 2097152) { // 2MB limit
                        alert(`File ${file.name} is too large. Maximum size is 2MB.`);
                        return;
                    }
                    
                    const reader = new FileReader();
                    
                    reader.onload = function(e) {
                        const div = document.createElement('div');
                        div.className = 'image-preview';
                        div.innerHTML = `
                            <img src="${e.target.result}" alt="Preview">
                            <span class="remove-image" onclick="removeImagePreview(${index})">×</span>
                        `;
                        previewContainer.appendChild(div);
                    }
                    
                    reader.readAsDataURL(file);
                });
            }
        }
        
        function removeImagePreview(index) {
            // Create a new DataTransfer object
            const dt = new DataTransfer();
            const input = document.querySelector('input[name="product_images[]"]');
            
            // Add all files except the one to remove
            for (let i = 0; i < input.files.length; i++) {
                if (i !== index) {
                    dt.items.add(input.files[i]);
                }
            }
            
            // Update the files in the input
            input.files = dt.files;
            
            // Refresh the preview
            previewImages(input);
        }
    </script>
</body>

</html>